 /*
  * main.c
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */


#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <syslog.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <libgen.h>     // dirname() and basename()
#include <ctype.h>      // isdigit()	  
#include <errno.h>
#include <dirent.h>
#include <blkid/blkid.h>
#include <pwd.h>
#include <grp.h>
#include <assert.h>
#include <stdbool.h>
#include <mntent.h>  // getmntent_r()

#include <ubus/libubus.h>
#include <libubox/blobmsg_json.h>

#include "def.h"
#include "init_list.h"
#include "sbuf.h"

#define MAX_SIZE 256

#define strdup_or_null( str )  (str) != NULL ? strdup(str) : NULL

const char *progname="usbmount";

static const char *pid_file="/var/run/usbmount/usbmount.pid";
static bool foreground = false;
static int pid_fd = -1;  // lock file

static struct ubus_context *ctx;
static struct ubus_request_data req_data;
static struct blob_buf bb_t;

void parse_args(int argc, char *argv[]);
pid_t get_pid_t(void);
int wait_on_kill(int s, int m);
void daemonize(void);

#define DIM(x) (sizeof(x)/sizeof(*(x)))

static const char     *sizes[]   = { "EiB", "PiB", "TiB", "GiB", "MiB", "KiB", "B" };
static const uint64_t  exbibytes = 1024ULL * 1024ULL * 1024ULL * 1024ULL * 1024ULL * 1024ULL;

// Usage: strncpy_t(str1, sizeof(str1), str2, strlen(str2));
// Copy string "in" with at most "insz" chars to buffer "out", which
// is "outsz" bytes long. The output is always 0-terminated. Unlike
// strncpy(), strncpy_t() does not zero fill remaining space in the
// output buffer:
static char *strncpy_t(char *out, size_t outsz, const char *in, size_t insz)
{
    assert(outsz > 0);
    while(--outsz > 0 && insz > 0 && *in) {
        *out++ = *in++;
        insz--;
    }
    *out = 0;
    return out;
}

// thread-safe and async-safe int to string for base 10
static void itoa10_safe(int val, char *str)
{
    int rc = 0, len = 0;
    int i = 0, j = 0;
    bool neg = false;
   
    // sign check
    if (val < 0) {
        val = -val;
        neg = true;
    }
   
    // consume, lowest-order to highest-order
    if (val == 0) {
        str[i] = '0';
        i++;
    }
    else {
        while (val > 0) {
            int r = val % 10;
            str[i] = '0' + r;
            i++;
            val /= 10;
        }
    }
   
    if (neg) {
        str[i] = '-';
        i++;
    }
   
    len = i;
    i--;
   
    // reverse order to get the number 
    while (j < i) {
        char tmp = *(str + i);
        *(str + i) = *(str + j);
        *(str + j) = tmp;
        j++;
        i--;
    }
   
    str[len] = '\0';
}

/**
 * \brief Create the directory to store the pidfile and the logfile.
 */
static void mkdir_piddir(const char *s)
{
    char *piddir, *tmp;

    piddir = strdup(s);
    tmp = strrchr(piddir, '/');
    if (tmp) {
        *tmp = '\0';
        mkdir(piddir, 0755);
    }

    free(piddir);
}
   
/**
 * \brief Convert a number of bytes into a file size
 */
static char *convert_size_into_string(uint64_t size)
{   
    uint64_t  multiplier = exbibytes;
    char *result = (char *) malloc(sizeof(char) * 20);
    if (! result) {
        fprintf(stderr, "%s: Memory allocation failure\n", progname);
        return NULL;
    }
    
    for (int i = 0; i < DIM(sizes); i++, multiplier /= 1024) {   
        if (size < multiplier)
            continue;
        if (size % multiplier == 0)
            sprintf(result, "%" PRIu64 " %s", size / multiplier, sizes[i]);
        else
            sprintf(result, "%.1f %s", (float) size / multiplier, sizes[i]);
        return result;
    }
    strcpy(result, "0");
    return result;
}

/**
 * \brief A subprogram like popen() but with diferences explained below
 * The pipe is to read in the current process and connected to stderr in the
 * new one. The intent is to report error messages. Plus pid is returned.
 * Disclaimer: The caller must close the pipe after use
 */
static FILE *epopen(const char *cmd,  pid_t *pid)
{
    int pipefd[2];
    pid_t p_id;
    FILE *pf;
    int rc;

    // Create the pipe
    if (pipe(pipefd)) {
        *pid = -1;
        return NULL;
    }

    p_id = fork();
 
    switch (p_id) {
    case 0: /* child */
        close(pipefd[0]);
        close(STDERR_FILENO);
        dup2(pipefd[1], STDERR_FILENO);
        close(pipefd[1]);
        execl("/bin/sh", "sh", "-c", cmd, (char *)NULL);
        fprintf(stderr, "%s: error invoking /bin/sh: %s\n", progname, strerror(errno));
        exit(EXIT_FAILURE); /* reminder: we are the child */
    
    case -1: /* fork() failed ! */
        rc = errno;
        close(pipefd[0]);
        close(pipefd[1]);
        errno = rc;
        *pid = -1;
        return NULL;
    
    default: /* the parent */
        close(pipefd[1]);
        pf = fdopen(pipefd[0], "r");
        *pid = p_id;
        return pf;
    }
}

enum {
    MOUNT_ID,
    MOUNT_UID,
    MOUNT_GID,
    MOUNT_DEVICE,
    MOUNT_MTPT,
    MOUNT_LABEL,
    __MOUNT_MAX,
};
 
static const struct blobmsg_policy mount_policy[] = {

    [MOUNT_ID]     = { .name="id",         .type=BLOBMSG_TYPE_INT32  },
    [MOUNT_UID]    = { .name="uid",        .type=BLOBMSG_TYPE_INT32  },
    [MOUNT_GID]    = { .name="gid",        .type=BLOBMSG_TYPE_INT32  },
    [MOUNT_DEVICE] = { .name="device",     .type=BLOBMSG_TYPE_STRING },
    [MOUNT_MTPT]   = { .name="mountpoint", .type=BLOBMSG_TYPE_STRING },
    [MOUNT_LABEL]  = { .name="label",      .type=BLOBMSG_TYPE_STRING },
};
 
enum {
    UMOUNT_ID,
    UMOUNT_UID,
    UMOUNT_GID,
    UMOUNT_DEVICE,
    __UMOUNT_MAX,
};
 
static const struct blobmsg_policy umount_policy[] = {
    
    [UMOUNT_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [UMOUNT_UID]    = { .name="uid",    .type=BLOBMSG_TYPE_INT32  },
    [UMOUNT_GID]    = { .name="gid",    .type=BLOBMSG_TYPE_INT32  },
    [UMOUNT_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};
 
enum {
    REMOVE_ID,
    REMOVE_DEVICE,
    __REMOVE_MAX,
};
 
static const struct blobmsg_policy remove_policy[] = {
    
    [REMOVE_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [REMOVE_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};

enum {
    GET_LABEL_ID,
    GET_LABEL_DEVICE,
    __GET_LABEL_MAX,
};
 
static const struct blobmsg_policy get_label_policy[] = {
    
    [GET_LABEL_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [GET_LABEL_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};

enum {
    GET_UUID_ID,
    GET_UUID_DEVICE,
    __GET_UUID_MAX,
};
 
static const struct blobmsg_policy get_uuid_policy[] = {
    
    [GET_UUID_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [GET_UUID_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};

enum {
    GET_FSTYPE_ID,
    GET_FSTYPE_DEVICE,
    __GET_FSTYPE_MAX,
};
 
static const struct blobmsg_policy get_fstype_policy[] = {
    
    [GET_FSTYPE_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [GET_FSTYPE_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};

enum {
    GET_SIZE_ID,
    GET_SIZE_DEVICE,
    __GET_SIZE_MAX,
};
 
static const struct blobmsg_policy get_size_policy[] = {
    
    [GET_SIZE_ID]     = { .name="id",     .type=BLOBMSG_TYPE_INT32  },
    [GET_SIZE_DEVICE] = { .name="device", .type=BLOBMSG_TYPE_STRING },
};

enum {
    INSTALL_SUBFOLDER_ID,
    INSTALL_SUBFOLDER_NAME,
    __INSTALL_SUBFOLDER_MAX,
};
 
static const struct blobmsg_policy install_subfolder_policy[] = {
    
    [INSTALL_SUBFOLDER_ID]   = { .name="id",   .type=BLOBMSG_TYPE_INT32  },
    [INSTALL_SUBFOLDER_NAME] = { .name="name", .type=BLOBMSG_TYPE_STRING },
};

static int mount_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    int rc;
    sbuf_t cmd;
    sbuf_t target;
    const char *name = NULL;
    int new_uid, new_gid;
    FILE *pfin = NULL;
    FILE *fp = NULL;
    pid_t pid;
    int wstatus;
    char cmdbuf[1024];
    char uuid[1024] = {0};
    const char *label = NULL;
    const char *mtpt = NULL;
    char uid_str[10];
    char gid_str[10];

    uid_str[9] = 0;
    gid_str[9] = 0;

    struct blob_attr *tb[__MOUNT_MAX];

    blobmsg_parse(mount_policy, ARRAY_SIZE(mount_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[MOUNT_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[MOUNT_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[MOUNT_DEVICE]);
    
    sbuf_init(&cmd);
    
    /* is it really unmounted? */
    sbuf_addstr(&cmd, "/bin/findmnt -n /dev");
    sbuf_addstr(&cmd, name);
    pfin = epopen(cmd.buf, &pid);
    if (pfin) { 
       fclose(pfin);
       waitpid(pid, &wstatus, 0);
    }
    sbuf_free(&cmd);
    
    if (wstatus == 0)
        goto _done;
    
    label = blobmsg_data(tb[MOUNT_LABEL]);
        
    new_uid = blobmsg_get_u32(tb[MOUNT_UID]);
    new_gid = blobmsg_get_u32(tb[MOUNT_GID]);
    
    mtpt = blobmsg_data(tb[MOUNT_MTPT]);
    
    rc = mkdir(mtpt, 0755);
    if (rc == -1) {
        switch (errno) {
        case EACCES :
            log_error("%s: the parent directory does not allow write\n", progname);
            exit(EXIT_FAILURE);
        case EEXIST:
            log_info("%s: pathname already exists\n", progname);
            break;
        case ENAMETOOLONG:
            log_error("%s: pathname is too long\n", progname);
            exit(EXIT_FAILURE);
        default:
            perror("mkdir");
            exit(EXIT_FAILURE);
        }
    }
    
    sbuf_init(&target);
    sbuf_concat(&target, 2, mtpt, "/");       
    rc = mkdir(target.buf, 0755);
    if (rc == -1) {
        switch (errno) {
        case EACCES :
            log_error("%s: parent directory does not allow write\n", progname);
            exit(EXIT_FAILURE);
        case EEXIST:
            log_info("%s: pathname already exists\n", progname);
            break;
        case ENAMETOOLONG:
            log_error("%s: pathname is too long\n", progname);
            exit(EXIT_FAILURE);
        default:
            perror("mkdir");
            exit(EXIT_FAILURE);
        }
    }
    
    // Reminder: usbmount passes $LABEL (devname) as an argument
    // If it's not labeled, then we get " (devname)", hence label[0] == ' '
    if (label && label[0] != ' ') {
        sbuf_addstr(&target, label);
    }
    else {
        /*
         * Example:  /sbin/blkid | grep "^/dev/sdb2" | awk -F'UUID=' '{print $2}' | cut -d'"' -f2
         * */
        sbuf_init(&cmd); 
        sbuf_concat(&cmd, 3, "/sbin/blkid | /bin/grep ^/dev/", name, 
                                " | /usr/bin/awk -F'UUID=' '{print $2}' | cut -d'\"' -f2");
        fp = popen(cmd.buf, "r");
        if (fp) {
            if (fgets(uuid, sizeof(uuid), fp))    
                uuid[strcspn(uuid, "\n")] = '\0';
            pclose(fp);
        }
        sbuf_free(&cmd);
        sbuf_addstr(&target, uuid);
    }
        
    rc = mkdir(target.buf, 0755);
    if (rc == -1) {
        switch (errno) {
        case EACCES :
            log_error("%s: parent directory does not allow write\n", progname);
            exit(EXIT_FAILURE);
        case EEXIST:
            log_info("%s: pathname already exists\n", progname);
            break;
        case ENAMETOOLONG:
            log_error("%s: pathname is too long\n", progname);
            exit(EXIT_FAILURE);
        default:
            perror("mkdir");
            exit(EXIT_FAILURE);
        }
    }
    
    chown(target.buf, new_uid, new_gid);
    chmod(target.buf, S_IRWXU);
    
    itoa10_safe(new_uid, uid_str);    
    itoa10_safe(new_gid, gid_str);
    
    sbuf_init(&cmd);
    sbuf_concat(&cmd, 9, "/bin/mount -o uid=", uid_str, ",gid=", 
                        gid_str, " /dev/", name, " \"", target.buf, "\"");
    
    memset(&cmdbuf, 0, sizeof(cmdbuf));
    pfin = epopen(cmd.buf, &pid);
    if (pfin) {
       if (fgets(cmdbuf, sizeof(cmdbuf), pfin))    
          cmdbuf[strcspn(cmdbuf, "\n")] = '\0';
       fclose(pfin);
       waitpid(pid, &wstatus, 0);
    }
    sbuf_free(&cmd);
    
    // The above command may fail because not all the filesystems allow to force 
    // the owner (e.g. ext4), leading to WEXITSTATUS(rc) = 32.
    // Only filesystems which do not support Linux permissions (like fat) have an 
    // attribute for ownership/groupship: uid=value and gid=value.
    // See the 'mount' manpage.
    
    if (wstatus != 0) {
        sbuf_init(&cmd);
        seteuid(new_uid);
        setegid(new_gid);
        sbuf_concat(&cmd, 5, "/bin/mount --rw /dev/", name, " \"", target.buf, "\"");
       
        memset(&cmdbuf, 0, sizeof(cmdbuf));
        pfin = epopen(cmd.buf, &pid);
        if (pfin) {
            if (fgets(cmdbuf, sizeof(cmdbuf), pfin))    
                cmdbuf[strcspn(cmdbuf, "\n")] = '\0';
            fclose(pfin);
            waitpid(pid, &wstatus, 0);
        }
        sbuf_free(&cmd);
       
        seteuid(0);
        setegid(0);
    
        sbuf_init(&cmd);
        sbuf_concat(&cmd, 2, "/bin/findmnt -n -S /dev/", name);
        pfin = epopen(cmd.buf, &pid);
        if (pfin) { 
            fclose(pfin);
            waitpid(pid, &wstatus, 0);
        }
        sbuf_free(&cmd);
       
        if (wstatus != 0) {
            struct stat sb;
            if (stat(target.buf, &sb)) {
                fprintf(stderr, "cannot stat %s\n", target);
                exit(EXIT_FAILURE);
            }
          
            //struct passwd *pw = getpwuid((int)sb.st_uid);
            //struct group  *gr = getgrgid((int)sb.st_gid);
            //if(!pw && !gr)
            rmdir(target.buf);
        }

        sbuf_free(&target);
    }

_done:
    blob_buf_init(&bb_t, 0);
    
    if (wstatus == 0 && cmdbuf[0] == 0)
        blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded ", cmdbuf);    
    else if (wstatus == 0 && cmdbuf)
        blobmsg_add_string(&bb_t, "Server reply - WARNING ", cmdbuf);
    else
        if (wstatus != 0) blobmsg_add_string(&bb_t, "Server reply - ERROR ", cmdbuf);
    
    ubus_send_reply(ctx, req, bb_t.head);
    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
    
    return 0;
}

static int umount_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    sbuf_t cmd;
    const char *name = NULL;
    char line[300];
    FILE *pf = NULL;
    FILE *pfin = NULL;
    pid_t pid;
    int wstatus;
    char cmdbuf[1024] = {0};           
    char mntbuf[1024] = {0};
    struct mntent ment_buf;
    struct mntent* ment_ptr = NULL;   
    const char *uuid = NULL;
    blkid_probe pr = NULL;

    struct blob_attr *tb[__UMOUNT_MAX]; 

    blobmsg_parse(umount_policy, ARRAY_SIZE(umount_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[UMOUNT_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[UMOUNT_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[UMOUNT_DEVICE]);

    pr = blkid_new_probe_from_filename(blobmsg_data(tb[UMOUNT_DEVICE]));
    if (!pr) {
        log_error("%s: Failed to open %s", progname, name);
        exit(EXIT_FAILURE);
    }

    blkid_probe_enable_partitions(pr, true);
    blkid_do_fullprobe(pr);
   
    blkid_probe_lookup_value(pr, "UUID", &uuid, NULL);
    if(uuid) {
        size_t len; 
        char eol[1024] = {0};
        char line[1024] = {0};
      
        FILE *fp = fopen("/etc/fstab", "r");
        if(!fp) {
            fprintf(stderr, "%s: fopen(%s): %s\n", progname, strerror(errno));
        }
      
        strncpy_t(line, sizeof(line), "UUID=", strlen("UUID="));
        strcat(line, uuid);
        len = strlen(line);
  
        while(fgets(eol, sizeof(eol), fp)) {
            eol[strcspn(eol, "\n")] = '\0';
            if(!strncmp(line, eol, len)) {
                fclose(fp);
                return -1;
            }
        }
    
        fclose(fp);
    }

    blkid_free_probe(pr);
    
    /*--------------- open on first call, rewind on other calls ---------------*/
    pf = fopen("/proc/mounts", "r");
    if (! pf) {
        switch(errno) {
        case ENOMEM:
            fprintf(stderr, "%s error opening /proc/mounts: %s\n",  progname, strerror(errno));
            break;
        default:
            fprintf( stderr, "%s: %s: /proc/mounts.\n", progname, strerror(errno) );
            exit(EXIT_FAILURE);
        }
    }
    
    /* decode every line of /proc/mounts */
    while (ment_ptr = getmntent_r(pf, &ment_buf, mntbuf, 1024)) {
        char *partitionname = basename(ment_ptr->mnt_fsname);
        size_t len = strlen(partitionname);
        if (! strcmp(partitionname, name)) {
            struct stat sb;
            int uid, gid;
            char mountpoint[256]={0};
       
            if (stat(ment_ptr->mnt_dir, &sb)) {
                fprintf(stderr, "cannot stat %s\n", ment_ptr->mnt_dir);
                exit(EXIT_FAILURE);
            }
                
            uid = blobmsg_get_u32(tb[UMOUNT_UID]);
            gid = blobmsg_get_u32(tb[UMOUNT_GID]);
          
            /*
            // Go ahead only in case the mountpoint is owned by the regular user
            if (uid != (int)sb.st_uid || gid != (int)sb.st_gid) 
                return 2;
            * */  
            
            sbuf_init(&cmd);
            sbuf_concat(&cmd, 2, "umount /dev/", name);
            pfin = epopen(cmd.buf, &pid);
            if (pfin) {
                if (fgets(cmdbuf, sizeof(cmdbuf), pfin))    
                    cmdbuf[strcspn(cmdbuf, "\n")] = '\0';
                fclose(pfin);
                waitpid(pid, &wstatus, 0);
            }
            sbuf_free(&cmd);
            
            if (wstatus == 0) {
                rmdir(ment_ptr->mnt_dir);
            }
            else {
                sbuf_init(&cmd);
                sbuf_concat(&cmd, 2, "umount -l /dev/", name);
                pfin = epopen(cmd.buf, &pid);
                if (pfin) {
                    if (fgets(cmdbuf, sizeof(cmdbuf), pfin))    
                        cmdbuf[strcspn(cmdbuf, "\n")] = '\0';
                    fclose(pfin);
                    waitpid(pid, &wstatus, 0);
                }
                sbuf_free(&cmd);

                if (wstatus == 0)
                    rmdir(ment_ptr->mnt_dir);
            }
            break;
        }
    }
    fclose(pf);
    
    sbuf_init(&cmd);
    sbuf_concat(&cmd, 2, "/bin/findmnt -n -S /dev/", name);
    pfin = epopen(cmd.buf, &pid);
    if (pfin) { 
       fclose(pfin);
       waitpid(pid, &wstatus, 0);
    }
    sbuf_free(&cmd);
    
    blob_buf_init(&bb_t, 0);

    if (wstatus == 0)
        blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded ", cmdbuf);
    else if (cmdbuf[0] != 0)
        blobmsg_add_string(&bb_t, "Server reply - ERROR ", cmdbuf);

    ubus_send_reply(ctx, req, bb_t.head);
    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
    
    return 0;
}

static int remove_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    sbuf_t cmd;
    char str[64] = {0};
    struct nodeList *l = NULL;
    char line[300];
    FILE *pf = NULL;
    bool unmounted = true;
    size_t len = 0;
    const char *name = NULL;
    FILE *pfin = NULL;
    pid_t pid;
    int wstatus;
    char cmdbuf[1024] = {0};
    bool unmounted_ok = true;

    struct blob_attr *tb[__REMOVE_MAX]; 

    blobmsg_parse(remove_policy, ARRAY_SIZE(remove_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[REMOVE_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[REMOVE_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[REMOVE_DEVICE]);
    
    sbuf_init(&cmd);
    sbuf_concat(&cmd, 2, "/bin/findmnt -n -S /dev/", name);
    pfin = epopen(cmd.buf, &pid);
    if (pfin) {
       char eol[1024] = {0};    
       if (fgets(eol, sizeof(cmdbuf), pfin))    
          unmounted_ok = false;
       fclose(pfin);
       waitpid(pid, &wstatus, 0);
    }
    sbuf_free(&cmd);
       
    if (! unmounted_ok)
        return -1;  
    
    memset(&cmd, 0, sizeof(cmd));
   
    l = get_devices(l);
 
    sbuf_init(&cmd);

    struct nodeList *aux = l;

    while (aux) {
       char* tmp = rindex( aux->nodeT.sysfs_path, '/' ) + 1;
	   if (! strcmp(name, tmp)) {
		   strncpy_t(str, sizeof(str), name, strlen(name));
		   if (aux->nodeT.removable) {
			  sbuf_addstr(&cmd, "/usr/bin/eject /dev/");
			  if (! strcmp(aux->nodeT.devtype_str, "partition")) {
				 int j = 0;
				 while (! isdigit(str[j]))
                    j++; 
			     str[j] = '\0';
				 sbuf_addstr(&cmd, str);
              }
              else if (! strcmp(aux->nodeT.devtype_str, "disk")) {
                 sbuf_addstr(&cmd, str);
              }
	       }
           else {
			  sbuf_addstr(&cmd, "echo 1 | tee /sys/block/");
			  if (! strcmp(aux->nodeT.devtype_str, "partition")) {
				 int j = 0;
				 while (! isdigit(str[j]))
                    j++; 
			     str[j] = '\0';
                 sbuf_concat(&cmd, 2, str, "/device/delete");
              }
              else if (! strcmp(aux->nodeT.devtype_str, "disk")) {
                 sbuf_concat(&cmd, 2, str, "/device/delete");
              }
           }
           break;
       }
       
       aux = aux->next;
    }
            
    if (l) 
       l = clearList(l);

    pfin = epopen(cmd.buf, &pid);
    if (pfin) {
       if (fgets(cmdbuf, sizeof(cmdbuf), pfin))    
          cmdbuf[strcspn(cmdbuf, "\n")] = '\0';
       fclose(pfin);
       waitpid(pid, &wstatus, 0);
    }
    
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t,"Server reply - Request has been proceeded: ", cmd.buf);
    sbuf_free(&cmd);
          
    ubus_send_reply(ctx, req, bb_t.head);
    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
    
    return 0;
}

static int get_label_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    char msg[64] = {0};
    sbuf_t devname;
    const char *name = NULL;
    const char *label = NULL;
    blkid_probe pr = NULL; 

    struct blob_attr *tb[__GET_LABEL_MAX];

    blobmsg_parse(get_label_policy, ARRAY_SIZE(get_label_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[GET_LABEL_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[GET_LABEL_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[GET_LABEL_DEVICE]);

    sbuf_init(&devname);
    sbuf_concat(&devname, 2, "/dev/", name);
    pr = blkid_new_probe_from_filename(devname.buf);
    if (! pr) {
        log_error("%s: Failed to open %s", progname, name);
        return -2;
    }
    sbuf_free(&devname);

    blkid_do_probe(pr);
    blkid_probe_lookup_value(pr, "LABEL", &label, NULL);
    
    if (label)
        strncpy_t(msg, sizeof(msg), label, strlen(label));

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded: ", msg);
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    blkid_free_probe(pr);
    
    return 0;
}



/*
static int get_uuid_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
   char msg[256] = {0};
   const char *uuid = NULL; 
   char *device = NULL;
   int i, nparts;
   blkid_probe pr;
   blkid_partlist ls;
   blkid_parttable root_tab;
   char *parent = NULL;
   char *pn = NULL;
   char delim[2];

   struct blob_attr *tb[__GET_UUID_MAX];

   blobmsg_parse(get_uuid_policy, ARRAY_SIZE(get_uuid_policy), tb, blob_data(msg_t), blob_len(msg_t));
   device = strdup_or_null(blobmsg_data(tb[GET_UUID_DEVICE]));
   if(strncmp(device, "/dev/", 5)) return -1;
   
   i=0;
   while(i<strlen(device) && !isdigit(device[i])) i++;
   pn = rindex(blobmsg_data(tb[GET_UUID_DEVICE]), device[i]);
   delim[0] = device[i];
   delim[1] = '\0';
   parent = strtok(device, delim);

    pr = blkid_new_probe_from_filename(parent);
    if (!pr) {
        log_error("%s: faild to create a new libblkid probe", progname, parent);
        return -1;
    }
        
    // Binary interface
    ls = blkid_probe_get_partitions(pr);
    if (!ls) {
        log_error("%s: failed to read partitions\n", progname, parent);
        return -1;
    }

    // Print info about the primary (root) partition table
    root_tab = blkid_partlist_get_table(ls);
    if (!root_tab) {
        log_error("%s: does not contain any known partition table\n", progname, parent);
        return -1;
    }
    
    nparts = blkid_partlist_numof_partitions(ls);
    if (!nparts) {
        fprintf(stderr, "%s: blkid_partlist_numof_partitions()\n", progname, parent);
        return -1;
    }

    for (i = 0; i < nparts; i++) {
        char i_str[10];
        i_str[9] = 0;
        
        itoa10_safe(i, i_str);
        if(!strcmp(pn, i_str)) {
           blkid_partition par = blkid_partlist_get_partition(ls, i);
           blkid_parttable tab = blkid_partition_get_table(par);
           const char *p = blkid_partition_get_uuid(par);
           if(p) {
              uuid = blkid_partition_get_uuid(par);
              break;
           }    
        }   
   }

   strncpy_t(msg, sizeof(msg), parent, strlen(parent));
   strcat(msg, pn);
   strcat(msg, "|");
   if(uuid) strcat(msg, uuid);
   else strcat(msg, "*");

   blob_buf_init(&bb_t, 0);
   blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded: ", msg);
   ubus_send_reply(ctx, req, bb_t.head);

   ubus_defer_request(ctx, req, &req_data);
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   blkid_free_probe(pr);
    
   return 0;
}
* */

static int get_uuid_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    char msg[64] = {0};
    sbuf_t devname;
    const char *uuid = NULL;
    const char *name = NULL;
    blkid_probe pr = NULL;

    struct blob_attr *tb[__GET_UUID_MAX];

    blobmsg_parse(get_uuid_policy, ARRAY_SIZE(get_uuid_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[GET_UUID_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[GET_UUID_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[GET_UUID_DEVICE]);

    sbuf_init(&devname);
    sbuf_concat(&devname, 2, "/dev/", name);
    pr = blkid_new_probe_from_filename(devname.buf);
    if (! pr) {
       log_error("%s: Failed to open %s", progname, name);
       exit(EXIT_FAILURE);
    }
    sbuf_free(&devname);

    blkid_probe_enable_partitions(pr, true);
    blkid_do_fullprobe(pr);
   
    blkid_probe_lookup_value(pr, "UUID", &uuid, NULL);
    
    if (uuid)
        strncpy_t(msg, sizeof(msg), uuid, strlen(uuid));

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded: ", msg);
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    blkid_free_probe(pr);
    
    return 0;
}

static int get_fstype_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    char msg[64] = {0};
    sbuf_t devname;
    const char *fstype = NULL;
    const char *name = NULL;
    blkid_probe pr = NULL;

    struct blob_attr *tb[__GET_FSTYPE_MAX];

    blobmsg_parse(get_fstype_policy, ARRAY_SIZE(get_fstype_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[GET_FSTYPE_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[GET_FSTYPE_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[GET_FSTYPE_DEVICE]);

    sbuf_init(&devname);
    sbuf_concat(&devname, 2, "/dev/", name);
    pr = blkid_new_probe_from_filename(devname.buf);
    if (! pr) {
        log_error("%s: Failed to open %s", progname, name);
        exit(EXIT_FAILURE);
    }
    sbuf_free(&devname);

    blkid_probe_enable_partitions(pr, true);
    blkid_do_fullprobe(pr);
   
    blkid_probe_lookup_value(pr, "TYPE", &fstype, NULL);
    
    if (fstype)
        strncpy_t(msg, sizeof(msg), fstype, strlen(fstype));

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded: ", msg);
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    blkid_free_probe(pr);
    
    return 0;
}

static int get_size_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    char msg[64] = {0};
    char *tmp = NULL;
    char *size = NULL;
    const char *name = NULL;
    char parent[32] = {0};
    sbuf_t devname;
    char *partno = NULL;
    blkid_probe pr = NULL;
    blkid_partlist ls;
    blkid_partition par;
    int nparts, i = 0;
    DIR *dirp;
    struct dirent *E;
    int nP = 0;
    const char *dir = "/sys/block";

    struct blob_attr *tb[__GET_SIZE_MAX];

    blobmsg_parse(get_size_policy, ARRAY_SIZE(get_size_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (strstr(blobmsg_data(tb[GET_SIZE_DEVICE]), "/"))
        name = rindex(blobmsg_data(tb[GET_SIZE_DEVICE]), '/') + 1;
    else
        name = blobmsg_data(tb[GET_SIZE_DEVICE]);
        
    dirp = opendir(dir);
    if (! dirp)
        return -errno;

    while ((errno = 0, E = readdir(dirp))) {
        if (strstr(E->d_name, ".") || strstr(E->d_name, ".."))
            continue; 
        if (! strncmp(name, E->d_name, strlen(E->d_name))) {
            sbuf_t path;
            sbuf_init(&path);
            sbuf_concat(&path, 6, dir, "/", E->d_name, "/", name, "/partition");
                    
            // Does '/sys/block/{E->d_name}/name/partition' exist?
            if (access(path.buf, F_OK) == 0) {
                size_t len = strlen(E->d_name); 
                strncpy_t(parent, sizeof(parent), E->d_name, strlen(E->d_name));
                sbuf_free(&path);
                while (! isdigit(name[len]))
                    len++;
                nP = atoi(name + len);
                break;
            }
            sbuf_free(&path);
        }
    }
    closedir(dirp);
 
    sbuf_init(&devname);
    sbuf_concat(&devname, 2, "/dev/", parent);
    pr = blkid_new_probe_from_filename(devname.buf);
    if (! pr) {
        log_error("%s: Failed to open %s", progname, parent);
        exit(EXIT_FAILURE);
    }
    sbuf_free(&devname);

    blkid_probe_enable_partitions(pr, true);
    blkid_do_fullprobe(pr);
   
    ls = blkid_probe_get_partitions(pr);
    nparts = blkid_partlist_numof_partitions(ls);
  
    for (i = 0; i < nparts; i++) {
        blkid_partition par = blkid_partlist_get_partition(ls, i);
        if (blkid_partition_get_partno(par) != nP)
            continue;
        blkid_loff_t sz =  blkid_partition_get_size(par);
        size = convert_size_into_string(sz);
        if (! size) {
            blkid_free_probe(pr);
            exit(EXIT_FAILURE);
        }
        break;
    } 
    
    if (size)
        strncpy_t(msg, sizeof(msg), size, strlen(size));

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", msg);
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    free(size);
    blkid_free_probe(pr);
    
    return 0;
}

static int install_subfolder_handler(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg_t)
{
    char msg[64] = {0};
    const char *name = NULL;

    struct blob_attr *tb[__INSTALL_SUBFOLDER_MAX];

    blobmsg_parse(install_subfolder_policy, ARRAY_SIZE(install_subfolder_policy), tb, blob_data(msg_t), blob_len(msg_t));
    name = blobmsg_data(tb[INSTALL_SUBFOLDER_NAME]);
    
    if (!strcmp(name, "by-label")) {
        mkdir("/dev/disk/by-label", 0755);
        strncpy_t(msg, sizeof(msg), "Installed subfolder /dev/disk/by-label", 38);
    }

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded: ", msg);
    ubus_send_reply(ctx, req, bb_t.head);

    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
    
    return 0;
} 

static const struct ubus_method usbmount_methods[] =
{
    UBUS_METHOD("mount", mount_handler, mount_policy),
    UBUS_METHOD("umount", umount_handler, umount_policy),
    UBUS_METHOD("remove", remove_handler, remove_policy),
    UBUS_METHOD("get_label", get_label_handler, get_label_policy),
    UBUS_METHOD("get_uuid", get_uuid_handler, get_uuid_policy),
    UBUS_METHOD("get_fstype", get_fstype_handler, get_fstype_policy),
    UBUS_METHOD("get_size", get_size_handler, get_size_policy),
    UBUS_METHOD("install_subfolder", install_subfolder_handler, install_subfolder_policy),
};
 

static struct ubus_object_type usbmount_obj_type =
    UBUS_OBJECT_TYPE("ering.usbmount", usbmount_methods);
 

static struct ubus_object usbmount_obj = {
    .name = "ering.usbmount", 
    .type = &usbmount_obj_type,
    .methods = usbmount_methods,
    .n_methods = ARRAY_SIZE(usbmount_methods),
    //.path=
};


int main(int argc, char **argv)
{      
    int ret;
    const char *ubus_socket="/var/run/ubus/ubus.sock";
    struct sigaction sa;
    
    parse_args(argc, argv);
   
    if (foreground == 0)
        daemonize();

    openlog(argv[0], LOG_PID|LOG_CONS, LOG_DAEMON);
    syslog(LOG_INFO, "Started %s", progname);
    
    uloop_init();
 
    ctx=ubus_connect(ubus_socket);
    if (! ctx)
        exit(EXIT_FAILURE);
 
    ubus_add_uloop(ctx);

    ret=ubus_add_object(ctx, &usbmount_obj);
    if (ret != 0)
        goto _ubus_fail;
    
    uloop_run();
    uloop_done();
    
_ubus_fail:
    ubus_free(ctx);

    if (access( pid_file, F_OK ) == 0) {
        if (pid_fd != -1) {
            int rc = lockf(pid_fd, F_ULOCK, 0);
            if (rc != 0)
                exit(EXIT_FAILURE);
            close(pid_fd);
        }        
        unlink(pid_file);
    }
   
    syslog(LOG_INFO, "Stopped %s", progname);
    closelog();
   
    return 0;
}

void parse_args(int argc, char *argv[])
{
    static struct option long_options[] = {
       
        {"foreground",            no_argument, 0, 'f'},
        {"kill",                  no_argument, 0, 'k'},
        {"check-running",         no_argument, 0, 'c'},
        {0, 0, 0}
    };
   
    int value, option_index = 0;
    int _kill = 0, _check = 0;
    while ((value = getopt_long(argc, argv, "fkc", long_options, &option_index)) != -1) {
        switch (value) {
        case 0:
            if (long_options[option_index].flag)
                break;
        case 'f':
            foreground = true;
            break;
        case 'k':
            _kill = 1;
            break;
        case 'c':
            _check = 1;
            break;
        default:
            syslog(LOG_ERR, "Unknown parameter.");
            break;
        }
    }
  
    if (_kill) {
        int rc = 0;
        if ((rc = wait_on_kill(SIGTERM, 60)) < 0) {
            log_error("Failed to kill daemon with SIGTERM: %s\n", strerror(errno));
            exit(1);
        }
        exit(EXIT_SUCCESS);
    }
   
    if (_check) {
        pid_t pid = get_pid_t();
        if (pid == (pid_t) -1 || pid == 0) {
            // Do not print anything here. Empty output is used in snetaid.postinst script 
            // to check whether or not sysvinit is supervising snetaid
            // printf("%s not running.\n", progname);
            exit(EXIT_FAILURE);
        }
        else {
            printf("%s process running as pid %u.\n", progname, pid);
            exit(EXIT_SUCCESS);
        }
    }
}

/**
 * \brief Get the PID of the process.
 */
pid_t get_pid_t(void)
{
    static char txt[MAX_SIZE];
    int fd = -1;
    pid_t ret = (pid_t) -1, pid;
    ssize_t l;
    long lpid;
    char *e = NULL;
   
    if ((fd = open(pid_file, O_RDWR, 0644)) < 0) {
        if ((fd = open(pid_file, O_RDONLY, 0644)) < 0) {
            if (errno != ENOENT)
                log_error("Failed to open pidfile '%s': %s , errno=%d\n", pid_file, strerror(errno), errno);
            goto _finish;
        }
    }
   
    if ((l = read(fd, txt, MAX_SIZE-1)) < 0) {
        int saved_errno = errno;
        log_error("read(): %s , errno=%d", strerror(errno), errno);
        unlink(pid_file);
        errno = saved_errno;
        goto _finish;
    }
   
    txt[l] = 0;
    txt[strcspn(txt, "\r\n")] = 0;
   
    errno = 0;
    lpid = strtol(txt, &e, 10);
    pid = (pid_t) lpid;
   
    if (errno != 0 || !e || *e || (long) pid != lpid) {
        unlink(pid_file);
        errno = EINVAL;
        log_warn("PID file corrupt, removing. (%s) , errno=%d", pid_file, errno);
        goto _finish;
    }
   
    if (kill(pid, 0) != 0 && errno != EPERM) {
        int saved_errno = errno;
        log_warn("Process %lu died: %s; trying to remove PID file. (%s)", (unsigned long) pid, strerror(errno), pid_file);
        unlink(pid_file);
        errno = saved_errno;
        goto _finish;
    }
   
    ret = pid;

_finish:
    if (fd != -1) {
        int rc = 0;
        if ((rc=lockf(fd, F_ULOCK, 0)) < 0) {
            log_error("%s: Cannot unlock file\n", progname);
            exit(EXIT_FAILURE);
        }
        close(fd);
    }
   
    return ret;
}

/**
 * \brief Kill the daemon process.
 */
int wait_on_kill(int s, int m)
{
    pid_t pid;
    time_t t;

    if ((pid = get_pid_t()) < 0) 
        return -1;

    if (kill(pid, s) < 0)
        return -1;

    t = time(NULL) + m;

    for (;;) {
        int r;
        struct timeval tv = { 0, 100000 };

        if (time(NULL) > t) {
            errno = ETIME;
            return -1;
        }

        if ((r = kill(pid, 0)) < 0 && errno != ESRCH)
            return -1;

        if (r)
            return 0;

        if (select(0, NULL, NULL, NULL, &tv) < 0)
            return -1;
    }
}

/**
 * \brief This function will daemonize this app
 */
void daemonize()
{
    int ret;
    pid_t pid = 0;
    int fd;

    /* Fork off the parent process.
     * The first fork will change our pid
     * but the sid and pgid will be the
     * calling process
     */
    pid = fork();

    /* An error occurred */
    if (pid < 0) {
        exit(EXIT_FAILURE);
    }

    /* Fork off for the second time.
     * The magical double fork. We're the session
     * leader from the code above. Since only the
     * session leader can take control of a tty
     * we will fork and exit the session leader.
     * Once the fork is done below and we use
     * the child process we will ensure we're
     * not the session leader, thus, we cannot
     * take control of a tty.
     */
    if (pid > 0)
        exit(EXIT_SUCCESS);

    /* On success: The child process becomes session leader */
    if (setsid() < 0)
        exit(EXIT_FAILURE);

    /* Ignore signal sent from child to parent process */
    signal(SIGCHLD, SIG_IGN);

    /* Fork off for the second time*/
    pid = fork();

    /* An error occurred */
    if (pid < 0)
        exit(EXIT_FAILURE);

    /* Success: Let the parent terminate */
    if (pid > 0)
        exit(EXIT_SUCCESS);
   
    /* Set new file permissions */
    umask(0);

    /* Change the working directory to /tmp */
    /* or another appropriated directory */
    ret = chdir("/tmp");
    if(ret != 0)
        exit(EXIT_FAILURE);

    /* Close all open file descriptors */
    for (fd = sysconf(_SC_OPEN_MAX); fd > 0; fd--)
        close(fd);

    /* Reopen stdin (fd = 0), stdout (fd = 1), stderr (fd = 2) */
    stdin  = fopen("/dev/null", "r");
    stdout = fopen("/dev/null", "w+");
    stderr = fopen("/dev/null", "w+");
   
    /* Try to write PID of daemon to lockfile */
    if (pid_file) {
        ssize_t sz = 0;
        char mypid[10];
        mypid[9] = 0;

        mkdir_piddir(pid_file);
      
        pid_fd = open(pid_file, O_RDWR|O_CREAT, 0640);
        if (pid_fd < 0) { 
            log_error("%s: Cannot open pidfile: %s\n", progname, strerror(errno));
            exit(EXIT_FAILURE);
        }
      
        if (lockf(pid_fd, F_TLOCK, 0) < 0) {
            /* Can't lock file */
            exit(EXIT_FAILURE);
        }

        // Get current PID
        itoa10_safe((intmax_t) getpid(), mypid);
        sz = write(pid_fd, mypid, strlen(mypid));
        if (sz == -1) { 
            log_error("%s: Unable to write pid: %s\n", progname, strerror(errno));
            exit(EXIT_FAILURE);
        }
        else {  
            log_error("%s: Unable to write pid: %s\n", progname, strerror(errno));
            exit(EXIT_FAILURE);
        }
    }
}
